#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gpdef.h"
#include "gpstdio.h"
#include "gpstdlib.h"
#include "gpgraphic.h"
#include "gpfont.h"
#include "gpmem.h"
#include "gfx/gp32menu_nobmp.h"
#include "driver.h"
#include "gp32_mame.h"

static int game_num=1024;
static int game_num_avail=0;
static char game_avail[1024];
static int last_game_selected=0;

int gp32_freq = 133;
extern int gp32_frameskip;
extern int gp32_frameskip_auto;
int gp32_sound = 1;
static int save_config = 0;
int highscore_enabled = 0;

/* Load/Save options in SMC */
#define SAVEDATASTRUCTURE "gp32_freq=%d;gp32_frameskip=%d;gp32_frameskip_auto=%d;gp32_sound=%d;gp32_clock_cpu=%d;gp32_clock_sound=%d;highscore_enabled=%d;gp32_new_gfx_core=%d;gp32_triple_buffer=%d;gp32_gamma=%d;\0"
#define SAVEDATAFILEDEF	"gp:\\gpmm\\mamegp32\\cfg\\mame.cfg\0"
#define SAVEDATAFILECFG "gp:\\gpmm\\mamegp32\\cfg\\%s.cfg\0"

/* Over-ride default analogue sensitivity from "gp:\\gpmm\\mamegp32\\joy\\ROM_NAME.cfg" */
#define JOY_FILENAME "gp:\\gpmm\\mamegp32\\joy\\%s.joy\0"
#define JOY_STRUCTURE "x_sensitivity=%d;y_sensitivity=%d;x_reversed=%d;y_reversed=%d;\0"
int load_details = 0;
int x_sensitivity = 0;
int y_sensitivity = 0;
int x_reversed = 0;
int y_reversed = 0;
#define KEY_FILENAME "gp:\\gpmm\\mamegp32\\key\\%s.key\0"
#define KEY_STRUCTURE "LEFT=%d;RIGHT=%d;UP=%d;DOWN=%d;FIRE1=%d;FIRE2=%d;FIRE3=%d;FIRE4=%d;FIRE5=%d;FIRE6=%d;FIRE7=%d;FIRE8=%d;FIRE9=%d;FIRE10=%d;\0"
extern struct KeySettings *key_settings;

int gp32_clock_cpu=80; 			/* Clock option for CPU's */
int gp32_clock_sound=80; 		/* Clock option for Audio CPU's */
extern int gp32_triple_buffer;  	/* From video.c */
extern int gp32_new_gfx_core;		/* From video.c */
extern int stretch; 			/* From video.c: 0=None, 1=Normal, 2=Mix */
extern int gp32_fps_display;		/* From video.c: 0=Off, 1=On */

static unsigned long *gpPalette=(unsigned long *)0x14A00400; /* Pointer to Palette */ 

/* External "menu applications" will use these */
char menuApp[128];
char romName[128];


/* Initialize the list of available games in the SMC */
void game_list_init(void) {

	unsigned int c;
	char filename[128];
	char thisApp[128];
	int length = 1024;

	/* Try to load any passed in arguments */
	char* appPath = GpAppPathGet(&length);
	sscanf(appPath, "%s %s %s", thisApp, menuApp, romName);

	/* Check Available Games */
	for (c=0;(c<game_num && drivers[c]);c++) {
		
		sprintf(filename,"gp:\\gpmm\\mamegp32\\roms\\%s.zip\0",drivers[c]->name);
		if (gm_lstrlen(romName) > 0)
		{
			if( (strcmp(romName, drivers[c]->name) == 0) && gp32_fexists(filename) ) 
			{
				/* Only play this game */
				game_avail[c]=1;
				game_num_avail++;	
				return;
			}
		}
		else if (gp32_fexists(filename)) {
			game_avail[c]=1;
			game_num_avail++;	
			continue;
		}

		sprintf(filename,"gp:\\gpmm\\mamegp32\roms\\%s\0",drivers[c]->name);
		if (gm_lstrlen(romName) > 0)
		{
			if( gp32_fexists(romName) && (strcmp(romName, filename) == 0) ) 
			{
				/* Only play this game */
				game_avail[c]=1;
				game_num_avail++;	
				return;
			}
		}
		else if (gp32_fexists(filename)) {
			game_avail[c]=1;
			game_num_avail++;
			continue;	
		}

	}
}

static void draw_border(void)
{
	/*
	GpBitBlt( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,(unsigned char*)gp32menu,0,0,gp32menu_width,gp32menu_height);
	*/
	GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );
}

static void game_list_view(int *pos) {

	int i;
	int view_pos;
	int aux_pos=0;
	int screen_y = 38;
	int screen_x = 40;

	/* Draw the menu */
	draw_border();

	/* Check Limits */
	if (*pos>(game_num_avail-1))
		*pos=game_num_avail-1;
	if (*pos<0)
		*pos=0;
					   
	/* Set View Pos */
	if (*pos<5) {
		view_pos=0;
	} else {
		if (*pos>game_num_avail-7) {
			view_pos=game_num_avail-11;
			view_pos=(view_pos<0?0:view_pos);
		} else {
			view_pos=*pos-5;
		}
	}

	/* Show List */
	for (i=0;i<game_num;i++) {
		if (game_avail[i]==1) {
			if (aux_pos>=view_pos && aux_pos<=view_pos+10) {
				gp32_gamelist_text_out( screen_x, screen_y, (char *)drivers[i]->description );
				if (aux_pos==*pos) {
					gp32_gamelist_text_out( screen_x-10, screen_y,">" );
					gp32_gamelist_text_out( screen_x-13, screen_y,"-" );
				}
				screen_y+=15;
			}
			aux_pos++;
		}
	}

}

static void game_list_select (int index, char *game) {
	int i;
	int aux_pos=0;
	for (i=0;i<game_num;i++) {
		if (game_avail[i]==1) {
			if(aux_pos==index) {
				gm_strcpy(game,drivers[i]->name);
			}
			aux_pos++;
		   }
	}
}

static char *game_list_description (int index) {
	int i;
	int aux_pos=0;
	for (i=0;i<game_num;i++) {
		if (game_avail[i]==1) {
			if(aux_pos==index) {
				return((char *)drivers[i]->description);
			}
			aux_pos++;
		   }
	}
	return ((char *)0);
}

static int load_game_options(const char* cfgFile) {
	F_HANDLE fileHandle;
	ERR_CODE errCode;
	unsigned long dataSize;
	char dataFromFile[256];
	int details_loaded = 0;
	asm_memset(dataFromFile,0,256);

	if( GpFileOpen( cfgFile, OPEN_R, &fileHandle )==SM_OK )
	{
		GpFileGetSize ( cfgFile, &dataSize );
		errCode = GpFileRead( fileHandle, dataFromFile, dataSize, NULL);
		if( errCode==SM_OK )
		{
			sscanf( dataFromFile, SAVEDATASTRUCTURE, &gp32_freq, &gp32_frameskip , &gp32_frameskip_auto, &gp32_sound, &gp32_clock_cpu, &gp32_clock_sound, &highscore_enabled, &gp32_new_gfx_core, &gp32_triple_buffer);
			if (gp32_new_gfx_core&2)
				stretch=1;
			if (gp32_new_gfx_core&4)
				stretch=2;
			gp32_new_gfx_core=gp32_new_gfx_core&1;
			details_loaded = 1;
		}
		GpFileClose( fileHandle );
	}

	return details_loaded;
}

static void load_default_keys(void)
{
	key_settings->JOY_LEFT		= GPC_VK_LEFT;
	key_settings->JOY_RIGHT		= GPC_VK_RIGHT;
	key_settings->JOY_UP		= GPC_VK_UP;
	key_settings->JOY_DOWN		= GPC_VK_DOWN;
	key_settings->JOY_FIRE1		= GPC_VK_FA;
	key_settings->JOY_FIRE2		= GPC_VK_FB;
	key_settings->JOY_FIRE3		= GPC_VK_SELECT;
	key_settings->JOY_FIRE4		= GPC_VK_FL;
	key_settings->JOY_FIRE5		= GPC_VK_START;
	key_settings->JOY_FIRE6		= GPC_VK_FR;
	key_settings->JOY_FIRE7		= GPC_VK_SELECT;
	key_settings->JOY_FIRE8		= GPC_VK_FL;
	key_settings->JOY_FIRE9		= GPC_VK_START;
	key_settings->JOY_FIRE10	= GPC_VK_FR;
}

static void load_keysettings(char* game_name)
{
	F_HANDLE fileHandle;
	ERR_CODE errCode;
	unsigned long dataSize;
	char *dataFromFile;
	char filename[64];

	/* Load default keys, used if .key file is missing */
	load_default_keys();
	sprintf(filename, KEY_FILENAME, game_name);

	if( GpFileOpen( filename, OPEN_R, &fileHandle )==SM_OK )
	{
		GpFileGetSize ( filename, &dataSize );
		dataFromFile = gm_zi_malloc(dataSize);
		errCode = GpFileRead( fileHandle, dataFromFile, dataSize, NULL);
		if( errCode==SM_OK )
		{
			sscanf( dataFromFile, KEY_STRUCTURE, &key_settings->JOY_LEFT, &key_settings->JOY_RIGHT, &key_settings->JOY_UP, &key_settings->JOY_DOWN, &key_settings->JOY_FIRE1, &key_settings->JOY_FIRE2, &key_settings->JOY_FIRE3, &key_settings->JOY_FIRE4, &key_settings->JOY_FIRE5, &key_settings->JOY_FIRE6, &key_settings->JOY_FIRE7, &key_settings->JOY_FIRE8, &key_settings->JOY_FIRE9, &key_settings->JOY_FIRE10 );
		}

		gm_free(dataFromFile);
		GpFileClose( fileHandle );
	}
}

/* Load any "over-ride" sensitivity variables from gp:\\gpmm\\mamegp32\\joy\\rom_name.joy file*/
static void load_sensitivity(char* game_name)
{
	F_HANDLE fileHandle;
	ERR_CODE errCode;
	unsigned long dataSize;
	char *dataFromFile;
	char filename[64];

	sprintf(filename, JOY_FILENAME, game_name);

	if( GpFileOpen( filename, OPEN_R, &fileHandle )==SM_OK )
	{
		GpFileGetSize ( filename, &dataSize );
		dataFromFile = gm_zi_malloc(dataSize);
		errCode = GpFileRead( fileHandle, dataFromFile, dataSize, NULL);
		if( errCode==SM_OK )
		{
			sscanf( dataFromFile, JOY_STRUCTURE, &x_sensitivity, &y_sensitivity, &x_reversed, &y_reversed);
			if( (x_sensitivity < 0 || x_sensitivity > 255) ||
				(y_sensitivity < 0 || y_sensitivity > 255) )
			{
				/* Values are out of range, report error and continue with default values */
				load_details = 0;
				gp32_gamelist_zero();
				gp32_text_log("Values loaded from file...");
				gp32_text_log(filename);
				gp32_text_log("are outside the range 1-255:");
				gp32_text_log(dataFromFile);
				gp32_text_log(" ");
				gp32_text_log("Default values will be loaded");
				gp32_text_log("Press any key to continue.");
				while(GpKeyGet()!=GPC_VK_NONE) { }
				while(GpKeyGet()==GPC_VK_NONE) { }
			}
			else
			{
				/* Loaded joystick values are valid */
				/* These details will override default details in update_from_file() in inptport.c */
                		load_details = 1;
			}
		}
		else if( errCode==ERR_FILE_EXIST )
		{
			/* No configuration file, continue with default values */
			load_details = 0;
		}
		else
		{
			/* Some other problem, report the error */
			load_details = 0;
			gp32_gamelist_zero();
			gp32_text_log_fmt("ERR_CODE='%d' opening file:", errCode);
			gp32_text_log(filename);
			gp32_text_log(" ");
			gp32_text_log("Default values will be loaded");
			gp32_text_log("Press any key to continue.");
			while(GpKeyGet()!=GPC_VK_NONE) { }
			while(GpKeyGet()==GPC_VK_NONE) { }
		}
		gm_free(dataFromFile);
		GpFileClose( fileHandle );
	}
}

static void save_game_options(const char *cfgFile) {
	F_HANDLE fileHandle;
	char settingsToSave[256];
	asm_memset(settingsToSave,0,256);

	if (GpFileCreate(cfgFile, ALWAYS_CREATE, &fileHandle )==SM_OK) {
		GpFileClose( fileHandle );
		if (GpFileOpen(cfgFile, OPEN_W, &fileHandle )==SM_OK) {
			if (stretch==1)
				gp32_new_gfx_core=gp32_new_gfx_core|2;
			if (stretch==2)
				gp32_new_gfx_core=gp32_new_gfx_core|4;
			sprintf(settingsToSave, SAVEDATASTRUCTURE, gp32_freq, gp32_frameskip, gp32_frameskip_auto, gp32_sound, gp32_clock_cpu, gp32_clock_sound, highscore_enabled, gp32_new_gfx_core, gp32_triple_buffer);
			gp32_new_gfx_core=gp32_new_gfx_core&1;
			GpFileWrite(fileHandle, (void *)settingsToSave, gm_lstrlen(settingsToSave)+1);
			GpFileClose( fileHandle );
			GpFATUpdate("GP://");
		}			
	}
}

static int LaunchMenuApp(const char* lpszAppPath, const char* lpszRomName)
{
	char command[256];
	ERR_CODE result;
	F_HANDLE fHandle;

	unsigned long dwDummy;
	unsigned long dwHeader, dwFileLength, dwInfoLength;

	int    i, j, k;
	int    n_read, n_axf_ro, n_axf_rw;
	unsigned char *p_axf,  *p_axf_ro, *p_axf_rw;
	unsigned char *p_work, *p_temp;

	result = GpFileOpen(lpszAppPath, OPEN_R, &fHandle);
	if (result != SM_OK)
		return -2;

	GpFileRead(fHandle, &dwHeader,     sizeof(unsigned long), &dwDummy);
	GpFileRead(fHandle, &dwFileLength, sizeof(unsigned long), &dwDummy);
	GpFileRead(fHandle, &dwInfoLength, sizeof(unsigned long), &dwDummy);

	GpFileSeek(fHandle, FROM_BEGIN, dwInfoLength + 8, (long *) &dwDummy);

	GpFileRead(fHandle, &n_axf_ro, sizeof(int), &dwDummy);
	GpFileRead(fHandle, &n_axf_rw, sizeof(int), &dwDummy);

	n_read = n_axf_ro + n_axf_rw;
	p_axf = (unsigned char*) gm_malloc(n_read + 4);
	if (!p_axf)
	{
		GpFileClose(fHandle);
		return -1;
	}

	p_axf_ro = p_axf + 4;
	p_axf_rw = p_axf_ro + n_axf_ro;
	*(int*) p_axf = n_read;

	GpFileRead(fHandle, p_axf_rw, n_axf_rw, &dwDummy);

	j = n_axf_rw;
	p_work = p_axf_rw;
	for (i = 0; i < j; i ++)
	{
		*p_work ^= 0xff;
		p_work ++;
	}


	GpFileRead(fHandle, p_axf_ro, n_axf_ro, &dwDummy);
	GpFileClose(fHandle);

	j = n_axf_ro;
	k = n_axf_rw;
	p_work = p_axf_ro;
	p_temp = p_axf_rw;
	for (i = 0; i < j; i ++)
	{
		*p_work ^= *p_temp;
		p_work ++;
		p_temp ++;
		if (p_temp >= (p_axf_rw + k))
		p_temp = p_axf_rw;
	}

	p_work = (unsigned char*) lpszAppPath;
	while (*p_work)
		p_work ++;
	while (*p_work != '.')
		p_work --;
	*p_work = 0;


	GpClockSpeedChange (67800000, 0x69032, 3);
	
	gm_strcpy( command, lpszAppPath );
	gm_strcat( command, " " );
	gm_strcat( command, lpszRomName );
	
	return GpAppExecute((char*) p_axf, (const char*) command);
}

static void end_mame(void)
{
 	GpClockSpeedChange(67800000, 0x69032, 3);

	if( gm_lstrlen(romName) == 0 )
	{
		__asm("swi #4;");
	}
	else
	{
		LaunchMenuApp(menuApp, romName);
		/* If LaunchMenuApp doesn't work, reboot the GP32 */
		__asm("swi #4;");
	}
}

static void get_frequency(int getNext)
{
	switch (gp32_freq) {
		case 133: gp32_freq = getNext ? 144 : 256; break;
		case 144: gp32_freq = getNext ? 150 : 133; break;
		case 150: gp32_freq = getNext ? 156 : 144; break;
		case 156: gp32_freq = getNext ? 160 : 150; break;
		case 160: gp32_freq = getNext ? 166 : 156; break;
		case 166: gp32_freq = getNext ? 168 : 160; break;
		case 168: gp32_freq = getNext ? 172 : 166; break;
		case 172: gp32_freq = getNext ? 176 : 168; break;
		case 176: gp32_freq = getNext ? 180 : 172; break;
		case 180: gp32_freq = getNext ? 184 : 176; break;
		case 184: gp32_freq = getNext ? 188 : 180; break;
		case 188: gp32_freq = getNext ? 192 : 184; break;
		case 192: gp32_freq = getNext ? 196 : 188; break;
		case 196: gp32_freq = getNext ? 200 : 192; break;
		case 200: gp32_freq = getNext ? 204 : 196; break;
		case 204: gp32_freq = getNext ? 208 : 200; break;
		case 208: gp32_freq = getNext ? 212 : 204; break;
		case 212: gp32_freq = getNext ? 216 : 208; break;
		case 216: gp32_freq = getNext ? 220 : 212; break;
		case 220: gp32_freq = getNext ? 224 : 216; break;
		case 224: gp32_freq = getNext ? 228 : 220; break;
		case 228: gp32_freq = getNext ? 232 : 224; break;
		case 232: gp32_freq = getNext ? 236 : 228; break;
		case 236: gp32_freq = getNext ? 240 : 232; break;
		case 240: gp32_freq = getNext ? 244 : 236; break;
		case 244: gp32_freq = getNext ? 248 : 240; break;
		case 248: gp32_freq = getNext ? 252 : 244; break;
		case 252: gp32_freq = getNext ? 256 : 248; break;
		case 256: gp32_freq = getNext ? 133 : 252; break;
	}
}

static int show_options(void)
{
	int ExKey=0, selected_option=0;
	int x_Pos = 40;	/* starting x position */
	int y_Pos = 32; /* starting y position */
	int line_size = 12;
	int half_line_size = line_size>>1;
	int options_count = 11;
	char* options[] = 
					{	"Frequency     %d%s",
						"Graphics Core %s",
						"Screen Fit    %s",
						"Frame-Skip    %s %d %s",
						"FPS Display   %s",
						"Buffering     %s",
						"Video Clock   %d%%",
						"Sound         %s",
						"Audio Clock   %d%%",
						"High Scores   %s",
						"Configuration %s"};

	while(1)
	{
		nflip++;
		nflip&=3;
		GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );

		/* Draw the background */
		/*GpBitBlt( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,(unsigned char*)gp32menu,0,0,gp32menu_width,gp32menu_height); */
		GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );

		/* Draw the options */
		/* gp32_gamelist_text_out(x_Pos,38,"Selected Game:\0"); */
		gp32_gamelist_text_out(x_Pos,y_Pos+half_line_size,game_list_description(last_game_selected));

		/* Frequency */
		if(gp32_freq > 200){
			gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*2),options[0],gp32_freq,"(Crazy!)");
		}
		else if(gp32_freq > 166){
			gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*2),options[0],gp32_freq,"(Warning!)");
		}
		else{
			gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*2),options[0],gp32_freq," MHz");
		}

		/* Graphics Core */
		switch (gp32_new_gfx_core) {
			case 0:	gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*3),options[1],"Old Core"); break;
			case 1:	gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*3),options[1],"New Core"); break;
		}
		
		/* Stretch */
		switch (stretch) {
			case 0: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*4),options[2],"Off"); break;
			case 1: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*4),options[2],"Skip Lines"); break;
			case 2: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*4),options[2],"Mix Lines"); break;
		}
		/* Frame-Skip */
		if(gp32_frameskip_auto && gp32_frameskip!=0) {
			gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*5),options[3],"<=",gp32_frameskip, "(Auto-skip)");
		}
		else{
			gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*5),options[3],"=",gp32_frameskip,"");
		}

		/* FPS Display */
		switch (gp32_fps_display) {
			case 0: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*6),options[4],"Off\0"); break;
			case 1: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*6),options[4],"On\0"); break;
		}

		/* Buffering */
		switch (gp32_triple_buffer) {
			case 0: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*7),options[5],"Normal\0"); break;
			case 1: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*7),options[5],"Triple Buffer\0"); break;
		}
		/* Video Clock */
		gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*8),options[6],gp32_clock_cpu);

		/* Sound */
		switch(gp32_sound)
		{
			case 0: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*9),options[7],"Not Emulated"); break;
			case 1: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*9),options[7],"Sound ON"); break;
			case 2: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*9),options[7],"Sound OFF"); break;
			case 3: gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*9),options[7],"Accurate"); break;
		}
		/* Audio Clock */
		gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*10),options[8],gp32_clock_sound);

		/* High Scores */
		switch (highscore_enabled) {
			case 0:	gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*11),options[9],"Don't Save Scores"); break;
			case 1:	gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*11),options[9],"Use High Scores"); break;
		}
		/* Save Configuration */
		switch (save_config) {
			case 0:	gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*12),options[10],"Don't Save Config"); break;
			case 1:	gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*12),options[10],"Save Game Config"); break;
			case 2:	gp32_gamelist_text_out_fmt(x_Pos,y_Pos+(line_size*12),options[10],"Save as Default"); break;
		}

		gp32_gamelist_text_out(x_Pos,y_Pos+half_line_size+(line_size*13),"Press A to confirm, B return\0");

		/* Show currently selected item */
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-10,y_Pos+((selected_option+2)*line_size),">", 254 );
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-13,y_Pos+((selected_option+2)*line_size),"-", 254 );

		GpSurfaceSet( &gpDraw[nflip] );
		while(GpKeyGet()!=GPC_VK_NONE) { }
		while((ExKey=GpKeyGet()) == GPC_VK_NONE) { }

		GpTextOut( NULL, &gpDraw[nflip],x_Pos-10,y_Pos+((selected_option+2)*line_size),">", 0 );
		GpTextOut( NULL, &gpDraw[nflip],x_Pos-13,y_Pos+((selected_option+2)*line_size),"-", 0 );

		if(ExKey==GPC_VK_DOWN){
			selected_option++;
			selected_option = selected_option % options_count;
		}
		else if(ExKey==GPC_VK_UP){
			selected_option--;
			if(selected_option<0)
				selected_option = options_count - 1;
		}
		else if(ExKey==GPC_VK_RIGHT || ExKey==GPC_VK_LEFT){
			switch(selected_option) {
			case 0:
				/* "Frequency" */
				get_frequency(ExKey==GPC_VK_RIGHT);
				break;
			case 1:
				/* "Graphics Core" */
				gp32_new_gfx_core=(gp32_new_gfx_core==0?1:0);
				break;
			case 2:
				/* "Stretch" */
				if(ExKey==GPC_VK_RIGHT){
					stretch++;
					if(stretch>2)
						stretch = 0;
				}
				else{
					stretch--;
					if(stretch<0)
						stretch = 2;
				}
				break;
			case 3:
				/* "Frame-Skip" */
				if(ExKey==GPC_VK_RIGHT){
					gp32_frameskip ++;
					if (gp32_frameskip > 5) {
						gp32_frameskip = 0;
						gp32_frameskip_auto=!gp32_frameskip_auto; 
					}
				}
				else{
					gp32_frameskip--;
					if (gp32_frameskip < 0){
						gp32_frameskip = 5;
						gp32_frameskip_auto=!gp32_frameskip_auto; 
					}
				}
				break;
			case 4:
				/* "FPS Display" */
				gp32_fps_display=(gp32_fps_display==0?1:0);
				break;
			case 5:
				/* "Buffering" */
				gp32_triple_buffer=(gp32_triple_buffer==0?1:0);
				break;
			case 6:
				/* "Video Clock" */
				if(ExKey==GPC_VK_RIGHT){
					gp32_clock_cpu += 10; /* Add 10% */
					if (gp32_clock_cpu > 200) /* 200% is the max */
						gp32_clock_cpu = 200;
				}
				else{
					gp32_clock_cpu -= 10; /* Subtract 10% */
					if (gp32_clock_cpu < 10) /* 10% is the min */
						gp32_clock_cpu = 10;
				}
				break;
			case 7:
				/* "Sound" */
				if(ExKey==GPC_VK_RIGHT){
					gp32_sound ++;
					if(gp32_sound > 3)
						gp32_sound = 0;
				}
				else{
					gp32_sound --;
					if(gp32_sound < 0)
                        			gp32_sound = 3;
				}
				break;
			case 8:
				/* "Audio Clock" */
				if(ExKey==GPC_VK_RIGHT){
					gp32_clock_sound += 10; /* Add 10% */
					if (gp32_clock_sound > 200) /* 200% is the max */
						gp32_clock_sound = 200;
				}
				else{
					gp32_clock_sound -= 10; /* Subtract 10% */
					if (gp32_clock_sound < 10) /* 10% is the min */
						gp32_clock_sound = 10;
				}
				break;
			case 9:
				/* "High Scores" */
				highscore_enabled=(highscore_enabled==0?1:0);
				break;
			case 10:
				/* "Save Configuration" */
				if(ExKey==GPC_VK_RIGHT){
					save_config ++;
					if(save_config>2)
						save_config = 0;
				}
				else{
					save_config--;
					if(save_config<0)
						save_config = 2;
				}
				break;
			}
		}
		if (ExKey & GPC_VK_FA){
			/* Selected games will be run */
			return 1;
		}
		else if(ExKey==GPC_VK_FB){
			if(gm_lstrlen(romName) > 0)
			{
				/* Return To "menu application" */
				end_mame();
			}
			else
			{
				/* Return To Menu */
				return 0;
			}
		}
	}
}

void select_game(char *argv[]) {

	int ExKey;
	int c;
	char romCfgFile[64];

	/* No Selected game */
	gm_strcpy(argv[1],"builtinn");

	/* Clean screen*/
	gp32_clear_screen();

	/* Wait until no key pressed */
	while(GpKeyGet()!=GPC_VK_NONE) { }

	/* Available games? */
	if(game_num_avail==0) {
		GpTextOut( NULL, &gpDraw[nflip],35,110,"ERROR: NO AVAILABLE GAMES FOUND", 254 );
		while(1)
		{
			while((ExKey=GpKeyGet())!=GPC_VK_NONE) {
				/* Reset GP32 */
				if ((ExKey & GPC_VK_FL) && (ExKey & GPC_VK_FR)) {
		 	 		end_mame();
				}
			}
			while((ExKey=GpKeyGet())==GPC_VK_NONE) {}

			/* Reset GP32 */
			if ((ExKey & GPC_VK_FL) && (ExKey & GPC_VK_FR)) {
	 	 		end_mame();
			}
		}
	}

	/* Set Palette */
	for (c=0;c<256;c++)
		gpPalette[c]=gp32menu_Pal[c];

	/* Wait until user selects a game */
	while(1) {
	/* Display games, if no rom name was passed to this application */
	if(gm_lstrlen(romName) <= 0)
	{
		nflip++;
		nflip&=3;
		GpRectFill( NULL, &gpDraw[nflip],0,0,gpDraw[nflip].buf_w,gpDraw[nflip].buf_h,0x0 );
		game_list_view(&last_game_selected);
		GpSurfaceSet( &gpDraw[nflip] );

		while((ExKey=GpKeyGet())!=GPC_VK_NONE) {
			/* Reset GP32 */
			if ((ExKey & GPC_VK_FL) && (ExKey & GPC_VK_FR)) {
		 	 	end_mame();
			}
		}
		while((ExKey=GpKeyGet())==GPC_VK_NONE) {}

		/* Reset GP32 */
		if ((ExKey & GPC_VK_FL) && (ExKey & GPC_VK_FR)) {
	 	 	end_mame();
		}
		if (ExKey & GPC_VK_UP) last_game_selected--;
		if (ExKey & GPC_VK_DOWN) last_game_selected++;
		if (ExKey & GPC_VK_FL) last_game_selected-=11;
		if (ExKey & GPC_VK_FR) last_game_selected+=11;
	}

		/* If Button A is pressed, or a rom name was passed to this application */
		if ((ExKey & GPC_VK_FA) || (gm_lstrlen(romName) > 0))
		{
			/* Select the game */
			game_list_select(last_game_selected, argv[1]);

			/* Try to load options from mamegp32\cfg\romname.CFG */
			sprintf(romCfgFile, SAVEDATAFILECFG, argv[1]);
			if( load_game_options(romCfgFile) == 0 )
			{
				/* No .cfg for this game, try to load from mame.cfg instead */
				load_game_options(SAVEDATAFILEDEF);
			}

			/* Emulation Options */
			if(show_options())
			{
				/* break out of the while(1) loop */
				break;
			}
		}
	}

	if( save_config==2 )
	{
		/* Update/Create the default mamegp32\cfg\mame.cfg AND mamegp32\cfg\rom_name.cfg file */
		save_game_options(SAVEDATAFILEDEF);
		sprintf(romCfgFile, SAVEDATAFILECFG, argv[1]);
		save_game_options(romCfgFile);
		save_config = 0;
	}
	else if ( save_config==1 )
	{
		/* Save everything to mamegp32\cfg\rom_name.cfg file */
		sprintf(romCfgFile, SAVEDATAFILECFG, argv[1]);
		save_game_options(romCfgFile);
		save_config = 0;
	}

	 /* Clean screen */
  	GpRectFill( NULL, &gpDraw[0],0,0,gpDraw[0].buf_w,gpDraw[0].buf_h,0x0 );
  	GpRectFill( NULL, &gpDraw[1],0,0,gpDraw[1].buf_w,gpDraw[1].buf_h,0x0 );

	/* Over-ride default key settings, used later in input.c */
	load_keysettings(argv[1]);

	/* Load any "over-ride" sensitivity variables from gp:\\gpmm\\mamegp32\\joy\\rom_name.joy file*/
	load_sensitivity(argv[1]);
}


/* Wait millisecs */
void Delay( int millisec )
{
	int nTick;
	nTick=GpTickCountGet();
	while( (GpTickCountGet()-nTick)<millisec );
}


void intro_screen(void) {

	int c;

	/* Clean screen*/
	gp32_clear_screen();
}
